/*
 * Isomorphic SmartGWT web presentation layer
 * Copyright 2000 and beyond Isomorphic Software, Inc.
 *
 * OWNERSHIP NOTICE
 * Isomorphic Software owns and reserves all rights not expressly granted in this source code,
 * including all intellectual property rights to the structure, sequence, and format of this code
 * and to all designs, interfaces, algorithms, schema, protocols, and inventions expressed herein.
 *
 *  If you have any questions, please email <sourcecode@isomorphic.com>.
 *
 *  This entire comment must accompany any portion of Isomorphic Software source code that is
 *  copied or moved from this file.
 */

package com.smartgwt.sample.showcase.client.shuttles;

import com.smartgwt.client.data.DataSource;
import com.smartgwt.client.data.DSCallback;
import com.smartgwt.client.data.DSRequest;
import com.smartgwt.client.data.DSResponse;
import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.widgets.layout.VLayout;
import com.smartgwt.client.widgets.form.DynamicForm;
import com.smartgwt.client.widgets.form.fields.TextItem;
import com.smartgwt.client.widgets.form.fields.ShuttleItem;
import com.smartgwt.client.widgets.form.fields.SubmitItem;
import com.smartgwt.client.widgets.form.fields.events.ClickEvent;
import com.smartgwt.client.widgets.form.fields.events.ClickHandler;
import com.smartgwt.client.widgets.grid.ListGrid;
import com.smartgwt.client.widgets.grid.ListGridRecord;
import com.smartgwt.client.widgets.grid.events.SelectionUpdatedEvent;
import com.smartgwt.client.widgets.grid.events.SelectionUpdatedHandler;
import com.smartgwt.client.types.SelectionStyle;
import com.smartgwt.sample.showcase.client.PanelFactory;
import com.smartgwt.sample.showcase.client.ShowcasePanel;
import com.smartgwt.sample.showcase.client.SourceEntity;

public class OneToManyShuttleSample extends ShowcasePanel {
    private static final String DESCRIPTION = 
        "This example demonstrates using a ShuttleItem to edit a one-to-many relationship "+
        "between two sets of data - Countries and Cities. Each Country contains multiple cities, "+
        "but each City can only be contained by one Country."+
        "<p>"+
        "Select a Country from the grid at the top to edit it in the form below. The ShuttleItem "+
        "allows you to add or remove cities from the country using a familiar shuttle style interface, "+
        "and then save the changes. Since the countries have a one-to-many relationship with their cities, "+
        "adding a city to one country will remove it from whichever country it was previously in."+
        "<p>"+
        "The data for this sample is stored in two database tables - a Country table and a City table, "+
        "where cities have a \"countryId\" field indicating which country they belong to."+
        "<p>"+
        "The Country dataSource has a \"cities\" field with multiple set to true and a foreignKey declaration "+
        "which maps back to the \"countryId\" on the City dataSource. This configuration of linked foreignKey "+
        "fields tell the SmartClient server that \"cities\" should be populated dynamically with the set of cities "+
        "associated with this country, based on the countryId values in the City database table."+
        "<p>"+
        "When the user makes changes to the \"cities\" field for some Country, the SmartClient server will update "+
        "countryId values for the appropriate records in the City database table.";
        
    public static class Factory implements PanelFactory {

        private String id;

        public ShowcasePanel create() {
            OneToManyShuttleSample panel = new OneToManyShuttleSample();
            id = panel.getID();
            return panel;
        }

        public String getID() {
            return id;
        }

        public String getDescription() {
            return DESCRIPTION;
        }
    }
    
    protected boolean isTopIntro() {
        return true;
    }

    public Canvas getViewPanel() {
        DataSource ds = DataSource.get("countrySQL_oneToManySample");
        ListGrid countriesGrid = new ListGrid();
        
        DynamicForm countryForm = new DynamicForm();
        countryForm.setDataSource(ds);
        countryForm.disable();
    	
        TextItem countryNameItem = new TextItem("countryName");
        ShuttleItem citiesItem = new ShuttleItem("cities");
        citiesItem.setCanEdit(true);
        citiesItem.setValueField("cityId");
        SubmitItem submitButtonItem = new SubmitItem();
        submitButtonItem.setTitle("Save");
        submitButtonItem.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                countryForm.saveData(new DSCallback() {
                    @Override
                    public void execute(DSResponse dsResponse, Object data, DSRequest dsRequest) {
                        countriesGrid.invalidateCache();
                    }
                });
            }
        });
    	
        countryForm.setFields(countryNameItem,citiesItem,submitButtonItem);

        countriesGrid.setDataSource(ds);
        countriesGrid.setAutoFetchData(true);
        countriesGrid.setHeight(200);
        countriesGrid.setShowResizeBar(true);
        countriesGrid.setSelectionType(SelectionStyle.SINGLE);
        countriesGrid.addSelectionUpdatedHandler(new SelectionUpdatedHandler() {
            @Override
            public void onSelectionUpdated(SelectionUpdatedEvent event) {
                ListGridRecord rec = countriesGrid.getSelectedRecord();
                if (rec == null) {
                    countryForm.clearValues();
                    countryForm.disable();
                } else {
                    countryForm.enable();
                    countryForm.editRecord(rec);
                }
            }
        });
    	
        VLayout vLayout = new VLayout();
        vLayout.setWidth100();
        vLayout.setHeight100();
        vLayout.setMembers(countriesGrid, countryForm);
        
        return vLayout;
    }

    public String getIntro() {
        return DESCRIPTION;
    }

    @Override
    public SourceEntity[] getSourceUrls() {
        return new SourceEntity[]{
            new SourceEntity("countrySQL_oneToManySample.ds.xml", XML, "source/countrySQL_oneToManySample.ds.xml.html", true),
            new SourceEntity("citySQL_oneToManySample.ds.xml", XML, "source/citySQL_oneToManySample.ds.xml.html", true)
        };
    }
    
}
